import processing.core.*; 
import processing.xml.*; 

import ddf.minim.*; 
import ddf.minim.signals.*; 
import ddf.minim.analysis.*; 
import ddf.minim.effects.*; 

import java.applet.*; 
import java.awt.Dimension; 
import java.awt.Frame; 
import java.awt.event.MouseEvent; 
import java.awt.event.KeyEvent; 
import java.awt.event.FocusEvent; 
import java.awt.Image; 
import java.io.*; 
import java.net.*; 
import java.text.*; 
import java.util.*; 
import java.util.zip.*; 
import java.util.regex.*; 

public class SmokeRings extends PApplet {






ArrayList<Ring> rings;
PGraphics lastFrame;
int feedbackOffset;

// band resolution determins how many rings and how many different frequencies each ring
// will represent.  a lower number = more rings
// 1 = 1 ring for each frequency in the spectrum
// 5 = 1 ring represents a range of 5 frequencies, the amplitude of that ring will be
// the peak of all frequencies in that part of the spectrum.
final int BAND_RESOLUTION = 5;

final float NORMAL_SCALE = 1.02f; // the zoom scale will try to settlg on this value
float myScale = 0.99f;
final float SCALE_PUSH = 0.005f; // when a new peak volume is found, the zoom scale pushes in this much
final float SCALE_FALLBACK = 0.001f; // this is the speed at which the zoom scale falls back to NORMAL_SCALE

float lastDrawTime=0; // keep track to how long it took to render each frame
float highestPeak = 0; // look for sudden increases in volume, this is the cue to do a zoom scale push

// the squench, if a ring's amplitude is below this level
// it will only draw as a dark grey ring
float MIN_AMP_TO_DRAW = 8.5f;

// how much the squench increases or decreases as the user
// presses up or down (respecively)
final float AMP_CHANGE = 0.1f;

Minim minim;
AudioPlayer player;
AudioInput input;
FFT fft;

boolean loadSuccessful = false;

public void setup()
{
  size(640,480,P3D);  
  smooth();
  noFill();
  
  // shift each pixel in the feedback down one row because they have a natural tendency to shift up one row
  feedbackOffset = (int)width; 
  
  minim = new Minim(this);
  try
  {
    player = minim.loadFile("song.mp3");
    input = minim.getLineIn();
    player.play();
    loadSuccessful=true;
    fft = new FFT(player.bufferSize(), player.sampleRate());
    int numRings = (int)(fft.specSize() / BAND_RESOLUTION);
  
    rings = new ArrayList<Ring>(numRings);
  
    for ( int i=0; i < numRings; i++ )
    {
      rings.add(new Ring());  
    }
  
  } catch (Exception ex)
  {    
  }
  

}

public void stop()
{
  player.close();
  input.close();
  minim.stop();
  super.stop();
}

public void keyPressed()
{
  if ( key == CODED )
  {
    if ( keyCode == UP )
      MIN_AMP_TO_DRAW += AMP_CHANGE;
    else if ( keyCode == DOWN )
      MIN_AMP_TO_DRAW -= AMP_CHANGE;      
    else if ( keyCode == LEFT )
      myScale += 5*SCALE_PUSH;
  }
  if ( MIN_AMP_TO_DRAW < 0 )
    MIN_AMP_TO_DRAW = 0;
  if ( MIN_AMP_TO_DRAW > 200 )
    MIN_AMP_TO_DRAW = 200;
  
}

public void draw()
{
  if ( !loadSuccessful )
  {
    background(0);
    fill(255);
    PFont font = createFont("Comic Sans", 32);
    textFont(font);
    textMode(SCREEN);
    text(
      "Please place desired song in app directory.\n" + 
      "Call it 'song.mp3'.\n" + 
      "Then run this application again.", 0, height/2);
    return;
  }
  
  float now = millis();
    
  if ( lastDrawTime > 0 && now - lastDrawTime > 30 )
  {
    // in danger of music skipping, drop a frame instead
    lastDrawTime=now;
    return;
  }
  lastDrawTime=now;
  
  background(0);
 
  SetScale();
  
  if ( lastFrame != null )
    image(lastFrame,0,0,width,height);
    
  float thisPeak = RenderRings();
  if ( thisPeak > highestPeak )
  {
    highestPeak = thisPeak;
    myScale += SCALE_PUSH;
  }
  else  
  {
    // we want the highest peak heard recently, not for the whole song...
    // so.. gradually "forget" the highest peak.
    highestPeak-=5;
  }
  
  if ( thisPeak < 1 && myScale > SCALE_FALLBACK )
  {
    // show's over folks!  Zoom out slowly so that all rings fall into the center
    myScale -= SCALE_FALLBACK;
  }
  
  // take a snapshot of this frame to use as a canvas for the next  
  lastFrame = Feedback(this.g);
}

public void SetScale()
{
  translate(width/2, height/2);
  scale(myScale);
  
  if ( myScale > NORMAL_SCALE )
    myScale -= SCALE_FALLBACK;
  
  translate(-width/2, -height/2);
}


public float RenderRings()
{
  float peakSum = 0;
  
  for ( int i=0; i < rings.size(); i++ )
  {
    Ring ring = rings.get(i);
    float peak = GetBandPeak(i);
    peakSum += peak;
    ring.Draw(peak);
  }  
  
  return peakSum;
}

public float GetBandPeak(int start)
{    
  fft.forward(player.mix);
  
  float peak = 0;
  
  for ( int i=start; i < start+BAND_RESOLUTION; i++ )
  {
    // introduce a bias the favors high freqs as they tend to be under powered
    float bias = pow(i+1, 1.1f) / (4*BAND_RESOLUTION);
    
    float thisValue = fft.getBand(i);
    if ( thisValue > 0 )
    {
      thisValue += bias ;
      if ( thisValue > peak )
        peak = thisValue;
    }
  }
  
  return peak;
}

public PGraphics Feedback(PGraphics source)
{
  PGraphics result = createGraphics(source.width, source.height, P2D);
  source.loadPixels();
  result.loadPixels();
 
  for ( int p = feedbackOffset; p < source.width*source.height; p++ )
  {
    result.pixels[p] = source.pixels[p-feedbackOffset];
  }
  
  result.updatePixels();
  
  return result;
}


class Vertex
{
  public float x,y,z;
  public Vertex()
  {
    x=y=z=0;
  }
  public Vertex(float _x, float _y, float _z)
  {
    x = _x;
    y = _y;
    z = _z;
  }
}

class Ring
{
  private Vertex myVertex;
  private float myRadius;
  private Colour myColor = new Colour();
  
  public Ring()
  {
    float edge = 100; // not too close to the edge
    myVertex = new Vertex( random(edge,width-edge), random(edge,height-edge), random(0,25) );
    myRadius = 1;
  }
  public void Draw(float amps)
  {
    boolean isOn=false;
    if ( amps >= MIN_AMP_TO_DRAW )
      isOn=true;
      
    float r = myRadius + amps + myVertex.z;

    float colorAmp = 1+(amps/15);
    
    if ( isOn )
    {
      stroke ( 
        min(myColor.r*colorAmp, 255),
        min(myColor.g*colorAmp, 255),
        min(myColor.b*colorAmp, 255) );
    }
    else    
    {
      stroke (20,20,20);
    }
    ellipse(myVertex.x, myVertex.y, r, r);

  }
}

class Colour
{
  public float r,g,b;
  public Colour()
  {
    r = random(32,64);
    g = random(32,64);
    b = random(32,64);
  }

}
  static public void main(String args[]) {
    PApplet.main(new String[] { "--bgcolor=#000000", "SmokeRings" });
  }
}
